package com.seatel.mobilehall.util.dialogannouncement

import android.app.Dialog
import android.content.Context
import android.content.DialogInterface
import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.view.View
import android.view.ViewGroup
import android.view.Window
import android.widget.Button
import android.widget.ImageView
import android.widget.TextView
import coil.load
import com.bumptech.glide.Glide
import com.seatel.mobilehall.R
import com.seatel.mobilehall.util.SeatelUtils.Companion.translatForDialog

class CustomDialog(
    context: Context,
    title: String,
    subtitle: String = "",
    imageUrl: String,
    image: Int,
    positiveButtonText: String = "",
    negativeButtonText: String = ""
) : Dialog(context) {

    private var positiveClickListener: DialogInterface.OnClickListener? = null
    private var negativeClickListener: DialogInterface.OnClickListener? = null

    init {
        initValentine(title, subtitle, imageUrl, image, positiveButtonText, negativeButtonText)
    }

    private fun initValentine(
        title: String,
        subtitle: String,
        imageUrl: String,
        image: Int,
        positiveButtonText: String,
        negativeButtonText: String
    ) {
        requestWindowFeature(Window.FEATURE_NO_TITLE)
        setContentView(R.layout.dialog_custom)
        setCancelable(false)
        window?.apply {
            setLayout(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT)
            attributes.windowAnimations = R.style.DialogAnimationScaleInOut
            setDimAmount(0.5f)
            setBackgroundDrawable(ColorDrawable(Color.TRANSPARENT))
        }

        val mIcon = findViewById<ImageView>(R.id.logo)
        val mTitle = findViewById<TextView>(R.id.title)
        val mDescription = findViewById<TextView>(R.id.description)
        val mBtnNegative = findViewById<Button>(R.id.btn_negative)
        val mBtnPositive = findViewById<Button>(R.id.btn_positive)

        mTitle?.text = translatForDialog(context, title)
        mDescription?.text = translatForDialog(context, subtitle)


        if (imageUrl.isNotEmpty() || imageUrl.isNotBlank()) {

            mIcon.load(imageUrl) {
                crossfade(true)
            }

        } else {
            Glide.with(context)
                .load(image)
                .placeholder(R.drawable.ic_logo)
                .into(mIcon)
        }



        mBtnPositive?.apply {
            text = translatForDialog(context, positiveButtonText)
            visibility = if (positiveButtonText.isNotEmpty()) View.VISIBLE else View.GONE
            setOnClickListener {
                positiveClickListener?.onClick(
                    this@CustomDialog,
                    DialogInterface.BUTTON_POSITIVE
                )
            }
        }

        mBtnNegative?.apply {
            text = translatForDialog(context, negativeButtonText)
            visibility = if (negativeButtonText.isNotEmpty()) View.VISIBLE else View.GONE
            setOnClickListener {
                negativeClickListener?.onClick(
                    this@CustomDialog,
                    DialogInterface.BUTTON_NEGATIVE
                )
            }
        }
    }

    fun setPositiveButton(
        text: String,
        clickListener: DialogInterface.OnClickListener
    ): CustomDialog {
        val mBtnPositive = findViewById<Button>(R.id.btn_positive)
        mBtnPositive?.text = translatForDialog(context, text)
        mBtnPositive?.visibility = View.VISIBLE
        positiveClickListener = clickListener
        return this
    }

    fun setNegativeButton(
        text: String,
        clickListener: DialogInterface.OnClickListener
    ): CustomDialog {
        val mBtnNegative = findViewById<Button>(R.id.btn_negative)
        mBtnNegative?.text = translatForDialog(context, text)
        mBtnNegative?.visibility = View.VISIBLE
        negativeClickListener = clickListener
        return this
    }
}
